package net.sf.twip.util;

import java.lang.annotation.Annotation;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

/**
 * In <code>java.lang.reflect</code> the parameters are not first class
 * citizens: you can only access them by several index methods, and the name is
 * not kept in the class file. This class tries to fill that gap as good as it
 * goes.
 */
public class Parameter {

	public static List<Parameter> of(Method method) {
		final List<Parameter> list = new ArrayList<Parameter>();
		for (int i = 0; i < method.getParameterTypes().length; i++) {
			list.add(new Parameter(method, i));
		}
		return Collections.unmodifiableList(list);
	}

	private final Method method;

	private final int index;

	public Parameter(Method method, int index) {
		this.method = method;
		this.index = index;
	}

	/**
	 * @return the annotation of that type or <code>null</code> if the parameter
	 *         is not annotated with that type.
	 */
	public <T extends Annotation> T getAnnotation(Class<T> annotationType) {
		for (final Annotation annotation : getAnnotations()) {
			if (annotationType.isInstance(annotation)) {
				return annotationType.cast(annotation);
			}
		}
		return null;
	}

	public Annotation[] getAnnotations() {
		return method.getParameterAnnotations()[index];
	}

	public int getIndex() {
		return index;
	}

	public Method getMethod() {
		return method;
	}

	public Class<?> getType() {
		return method.getParameterTypes()[index];
	}

	public <T extends Annotation> boolean isAnnotationPresent(Class<T> annotationType) {
		return getAnnotation(annotationType) != null;
	}

	@Override
	public String toString() {
		return Parameter.class.getSimpleName() + "#" + getIndex() + " of " + getMethod().toGenericString();
	}

}
