package net.sf.twip.parameterhandler;

import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import net.sf.twip.Assume;
import net.sf.twip.NotNull;
import net.sf.twip.Values;
import net.sf.twip.internal.TwipConfigurationErrorCallingMethod;
import net.sf.twip.internal.TwipConfigurationErrorGettingField;
import net.sf.twip.internal.TwipConfigurationErrorNoSuchMethod;
import net.sf.twip.internal.TwipConfigurationErrorNonPrimitive;
import net.sf.twip.internal.TwipConfigurationErrorNonStatic;
import net.sf.twip.internal.TwipConfigurationErrorNotArrayOrCollection;
import net.sf.twip.internal.TwipConfigurationErrorNullPointer;
import net.sf.twip.util.Parameter;

/**
 * Produces all the values to call for one {@link Parameter}, including only
 * those that match the {@link Assume} or {@link NotNull} annotations.
 * 
 * @see AbstractNumberParameterHandler for the expessions allowed for numbers
 */
public abstract class ParameterHandler {

	protected final Parameter parameter;

	public ParameterHandler(Parameter parameter) {
		this.parameter = parameter;
	}

	/**
	 * Get the values from the method or field specified in the {@link Values}
	 * annotation... with a lot of collecting useful error information.
	 */
	private Object[] getAnnotatedValues(Parameter parameter) {
		final String name = parameter.getAnnotation(Values.class).value();
		try {
			final Field field = parameter.getMethod().getDeclaringClass().getField(name);
			if (!Modifier.isStatic(field.getModifiers()))
				throw new TwipConfigurationErrorNonStatic("the field '" + name
						+ "' named in the @Values annotation of " + parameter + " must be static.");
			if (field.getType().isArray()) {
				if (field.getType().getComponentType().isPrimitive())
					throw new TwipConfigurationErrorNonPrimitive("the values of the field '" + name
							+ "' named in the @Values annotation of " + parameter
							+ " must not be primitives.");
				final Object[] result = (Object[]) field.get(null);
				if (result == null)
					throw new TwipConfigurationErrorNullPointer("the field '" + name
							+ "' named in the @Values annotation of " + parameter + " is null");
				return result;
			} else if (Collection.class.isAssignableFrom(field.getType())) {
				final Collection<?> result = (Collection<?>) field.get(null);
				if (result == null)
					throw new TwipConfigurationErrorNullPointer("the field '" + name
							+ "' named in the @Values annotation of " + parameter + " is null");
				return result.toArray();
			} else {
				throw new TwipConfigurationErrorNotArrayOrCollection("the field '" + name
						+ "' named in the @Values annotation of " + parameter
						+ " must be an array or collection.");
			}
		} catch (final IllegalAccessException e) {
			throw new TwipConfigurationErrorGettingField("can't access field '" + name
					+ "' named in the @Values annotation of " + parameter, e);
		} catch (final NoSuchFieldException ex) {
			try {
				final Method method = parameter.getMethod().getDeclaringClass().getMethod(name);
				if (!Modifier.isStatic(method.getModifiers()))
					throw new TwipConfigurationErrorNonStatic("the method '" + name
							+ "' named in the @Values annotation of " + parameter + " must be static.");
				if (method.getReturnType().isArray()) {
					if (method.getReturnType().getComponentType().isPrimitive())
						throw new TwipConfigurationErrorNonPrimitive("the values returned by the method '"
								+ name + "' named in the @Values annotation of " + parameter
								+ " must not be primitives.");
					final Object[] result = (Object[]) method.invoke(null);
					if (result == null)
						throw new TwipConfigurationErrorNullPointer("the method '" + name
								+ "' named in the @Values annotation of " + parameter + " returns null");
					return result;
				} else if (Collection.class.isAssignableFrom(method.getReturnType())) {
					final Collection<?> collection = (Collection<?>) method.invoke(null);
					if (collection == null)
						throw new TwipConfigurationErrorNullPointer("the method '" + name
								+ "' named in the @Values annotation of " + parameter + " returns null.");
					return collection.toArray();
				} else {
					throw new TwipConfigurationErrorNotArrayOrCollection("the method '" + name
							+ "' named in the @Values annotation of " + parameter
							+ " must return an array or collection.");
				}
			} catch (final IllegalAccessException e) {
				throw new TwipConfigurationErrorCallingMethod("can't access the method '" + name
						+ "' named in the @Values annotation of " + parameter, e);
			} catch (final NoSuchMethodException e) {
				throw new TwipConfigurationErrorNoSuchMethod("there is no method or field '" + name
						+ "' named in the @Values annotation of " + parameter, e);
			} catch (final InvocationTargetException e) {
				throw new TwipConfigurationErrorCallingMethod("can't invoke method '" + name
						+ "' named in the @Values annotation of " + parameter, e);
			}
		}
	}

	protected abstract Object[] getDefaultParameterValues();

	public Object[] getParameterValues() {
		final Object[] values = (parameter.isAnnotationPresent(Values.class)) ? getAnnotatedValues(parameter)
				: getDefaultParameterValues();
		if (!parameter.isAnnotationPresent(Assume.class))
			return values;
		final List<Object> result = new ArrayList<Object>();
		for (final Object value : values) {
			if (test(value)) {
				result.add(value);
			}
		}
		if (!parameter.isAnnotationPresent(NotNull.class) && !parameter.getType().isPrimitive()) {
			result.add(null);
		}
		return result.toArray();
	}

	/**
	 * Should this value go into the parameter values?
	 * 
	 * @param value
	 *            the value to check
	 */
	protected boolean test(Object value) {
		return true;
	}
}
