package net.sf.twip.parameterhandler;

import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.lang.reflect.ParameterizedType;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import net.sf.twip.AutoTwip;
import net.sf.twip.TwiP;
import net.sf.twip.internal.FrameworkMethodWithArgs;
import net.sf.twip.internal.TwipConfigurationErrorCallingMethod;
import net.sf.twip.internal.TwipConfigurationErrorDuplicateAutoTwip;
import net.sf.twip.internal.TwipConfigurationErrorGettingField;
import net.sf.twip.internal.TwipConfigurationErrorNonStatic;
import net.sf.twip.internal.TwipConfigurationErrorNotArrayOrCollection;
import net.sf.twip.internal.TwipConfigurationErrorNullPointer;
import net.sf.twip.internal.TwipConfigurationErrorUnknownParameterType;
import net.sf.twip.internal.TwipConfigurationErrorVoidAutoTwipMethod;
import net.sf.twip.util.Parameter;

public class AutoTwipParameterHandler extends ParameterHandler {

	private static abstract class AbstractAutoTwipMethodHandler implements AutoTwipHandler {
		private final Method method;

		public AbstractAutoTwipMethodHandler(Method method) {
			this.method = method;
		}

		public abstract void invoke(FrameworkMethodWithArgs method, List<Object> result) throws Throwable;

		private TwipConfigurationErrorCallingMethod newMethodCallingError(
				final FrameworkMethodWithArgs parameterizedMethod, final Throwable e) {
			return new TwipConfigurationErrorCallingMethod("can't initialize AutoTwip values from "
					+ parameterizedMethod, e);
		}

		public Object[] run() {
			final List<Object> result = new ArrayList<Object>();
			for (final FrameworkMethodWithArgs parameterizedMethod : TwiP
					.computeParameterizedTestMethods(method)) {
				try {
					invoke(parameterizedMethod, result);
				} catch (final ThreadDeath e) {
					throw e;
				} catch (final VirtualMachineError e) {
					throw e;
				} catch (final LinkageError e) {
					throw e;
				} catch (final Throwable e) { // NOSONAR
					throw newMethodCallingError(parameterizedMethod, e);
				}
			}
			return result.toArray();
		}
	}

	private static class AutoTwipFieldArrayHandler implements AutoTwipHandler {

		private final Field field;

		public AutoTwipFieldArrayHandler(Field field) {
			this.field = field;
		}

		public Object[] run() {
			try {
				return (Object[]) field.get(null);
			} catch (final Exception e) {
				throw new TwipConfigurationErrorGettingField("while calling AutoTwip field " + field, e);
			}
		}
	}

	private static class AutoTwipFieldCollectionHandler implements AutoTwipHandler {

		private final Field field;

		public AutoTwipFieldCollectionHandler(Field field) {
			this.field = field;
		}

		public Object[] run() {
			try {
				return ((Collection<?>) field.get(null)).toArray();
			} catch (final Exception e) {
				throw new TwipConfigurationErrorGettingField("while calling AutoTwip field " + field, e);
			}
		}
	}

	private interface AutoTwipHandler {
		public Object[] run();
	}

	private static class AutoTwipMethodArrayHandler extends AbstractAutoTwipMethodHandler {

		public AutoTwipMethodArrayHandler(Method method) {
			super(method);
		}

		@Override
		public void invoke(FrameworkMethodWithArgs method, List<Object> result) throws Throwable {
			for (final Object object : ((Object[]) method.invokeExplosively(null))) {
				result.add(object);
			}
		}
	}

	private static class AutoTwipMethodCollectionHandler extends AbstractAutoTwipMethodHandler {

		public AutoTwipMethodCollectionHandler(Method method) {
			super(method);
		}

		@Override
		public void invoke(FrameworkMethodWithArgs method, List<Object> result) throws Throwable {
			for (final Object object : ((Collection<?>) method.invokeExplosively(null))) {
				result.add(object);
			}
		}
	}

	private static class AutoTwipMethodResultHandler extends AbstractAutoTwipMethodHandler {

		public AutoTwipMethodResultHandler(Method method) {
			super(method);
		}

		@Override
		public void invoke(FrameworkMethodWithArgs method, List<Object> result) throws Throwable {
			final Object object = method.invokeExplosively(null);
			result.add(object);
		}
	}

	private final AutoTwipHandler autoTwipHandler;

	public AutoTwipParameterHandler(Parameter parameter) {
		super(parameter);
		this.autoTwipHandler = findAutoTwip(parameter);
	}

	private void fillAutoTwipFields(Map<Class<?>, AutoTwipHandler> handlers, Class<?> declaringClass) {
		for (final Field field : declaringClass.getDeclaredFields()) {
			if (field.isAnnotationPresent(AutoTwip.class)) {
				if (!Modifier.isStatic(field.getModifiers()))
					throw new TwipConfigurationErrorNonStatic("the field " + field
							+ " is annotated as @AutoTwip, but it is not static!");
				Class<?> type = field.getType();
				AutoTwipHandler handler;
				if (type.isArray()) {
					type = type.getComponentType();
					handler = new AutoTwipFieldArrayHandler(field);
				} else if (Collection.class.isAssignableFrom(type)) {
					final ParameterizedType parameterizedType = (ParameterizedType) field.getGenericType();
					type = (Class<?>) parameterizedType.getActualTypeArguments()[0];
					handler = new AutoTwipFieldCollectionHandler(field);
				} else {
					throw new TwipConfigurationErrorNotArrayOrCollection("the field " + field
							+ " is annotated as @AutoTwip, but it's not an array or collection!");

				}
				final AutoTwipHandler old = handlers.put(type, handler);
				if (old != null) {
					throw new TwipConfigurationErrorDuplicateAutoTwip("the field " + field
							+ " is annotated as @AutoTwip, but it's of the same type as the AutoTwip " + old);
				}
			}
		}
	}

	private void fillAutoTwipMethods(Map<Class<?>, AutoTwipHandler> handlers, Class<?> declaringClass) {
		for (final Method method : declaringClass.getDeclaredMethods()) {
			if (method.isAnnotationPresent(AutoTwip.class)) {
				if (!Modifier.isStatic(method.getModifiers()))
					throw new TwipConfigurationErrorNonStatic("the method " + method
							+ " is annotated as @AutoTwip, but it is not static!");
				Class<?> type = method.getReturnType();
				if (Void.TYPE.equals(type))
					throw new TwipConfigurationErrorVoidAutoTwipMethod("the method " + method
							+ " is annotated as @AutoTwip, but it returns void!");
				AutoTwipHandler handler;
				if (type.isArray()) {
					type = type.getComponentType();
					handler = new AutoTwipMethodArrayHandler(method);
				} else if (Collection.class.isAssignableFrom(type)) {
					final ParameterizedType parameterizedType = (ParameterizedType) method
							.getGenericReturnType();
					type = (Class<?>) parameterizedType.getActualTypeArguments()[0];
					handler = new AutoTwipMethodCollectionHandler(method);
				} else {
					handler = new AutoTwipMethodResultHandler(method);
				}
				final AutoTwipHandler old = handlers.put(type, handler);
				if (old != null) {
					throw new TwipConfigurationErrorDuplicateAutoTwip("the method " + method
							+ " is annotated as @AutoTwip, but it returns the same type as the AutoTwip "
							+ old);
				}
			}
		}
	}

	private AutoTwipHandler findAutoTwip(Parameter parameter) {
		final Map<Class<?>, AutoTwipHandler> handlers = findAutoTwipHandlers(parameter.getMethod()
				.getDeclaringClass());
		return handlers.get(parameter.getType());
	}

	private Map<Class<?>, AutoTwipHandler> findAutoTwipHandlers(final Class<?> declaringClass) {
		final Map<Class<?>, AutoTwipHandler> handlers = new HashMap<Class<?>, AutoTwipHandler>();
		fillAutoTwipMethods(handlers, declaringClass);
		fillAutoTwipFields(handlers, declaringClass);
		return handlers;
	}

	@Override
	protected Object[] getDefaultParameterValues() {
		if (autoTwipHandler == null)
			throw new TwipConfigurationErrorUnknownParameterType("TwiP can't produce parameters of type ["
					+ parameter.getType().getName() + "]. Please add a @Value annotation to parameter #"
					+ parameter.getIndex() + " of method " + parameter.getMethod()
					+ " or declare a @AutoTwip field or method.");
		final Object[] result = autoTwipHandler.run();
		if (result == null)
			throw new TwipConfigurationErrorNullPointer("The AutoTwip " + autoTwipHandler
					+ " is/returns null");
		return result;
	}
}
