package net.sf.twip.parameterhandler;

import java.util.regex.Matcher;
import java.util.regex.Pattern;

import net.sf.twip.Assume;
import net.sf.twip.internal.TwipConfigurationErrorIllegalAssumeExpression;
import net.sf.twip.util.Parameter;

/**
 * You can assume number parameters to be <code>&gt;</code>, <code>&gt;=</code>,
 * <code>!=</code>, <code>&lt;=</code>, or <code>&lt;</code> to a numeric
 * constant. You can assume float and double parameters additionally to be
 * <code>!= NaN</code>, <code>!= INF</code>, and <code>!= -INF</code>. Note that
 * <code>null</code> "slips through" these tests, i.e. if you assume that an
 * Integer argument is <code>&gt;= 0</code>, then <code>null</code> is still
 * passed in.
 */
abstract public class AbstractNumberParameterHandler extends ParameterHandler {

	private static final class EqualsTester implements Tester {
		private final Comparable<Object> comparable;

		private EqualsTester(Comparable<Object> comparable) {
			this.comparable = comparable;
		}

		public boolean test(Object value) {
			return value == null || 0 != comparable.compareTo(value);
		}
	}

	private static final class GreaterOrEqualsTester implements Tester {
		private final Comparable<Object> comparable;

		private GreaterOrEqualsTester(Comparable<Object> comparable) {
			this.comparable = comparable;
		}

		public boolean test(Object value) {
			return value == null || 0 >= comparable.compareTo(value);
		}
	}

	private static final class GreaterThanTester implements Tester {
		private final Comparable<Object> comparable;

		private GreaterThanTester(Comparable<Object> comparable) {
			this.comparable = comparable;
		}

		public boolean test(Object value) {
			return value == null || 0 > comparable.compareTo(value);
		}
	}

	private static final class LessOrEqualsTester implements Tester {
		private final Comparable<Object> comparable;

		private LessOrEqualsTester(Comparable<Object> comparable) {
			this.comparable = comparable;
		}

		public boolean test(Object value) {
			return value == null || 0 <= comparable.compareTo(value);
		}
	}

	private static final class LessThanTester implements Tester {
		private final Comparable<Object> comparable;

		private LessThanTester(Comparable<Object> comparable) {
			this.comparable = comparable;
		}

		public boolean test(Object value) {
			return value == null || 0 < comparable.compareTo(value);
		}
	}

	protected interface Tester {
		public boolean test(Object value);
	}

	private static final class TrueTester implements Tester {
		public boolean test(Object value) {
			return true;
		}
	}

	private static final Pattern EXPRESSION = Pattern.compile("(<|>|!)(=?)\\s*(-?\\w+)");

	private final Tester tester;

	public AbstractNumberParameterHandler(Parameter parameter) {
		super(parameter);
		this.tester = getTester();
	}

	protected abstract Comparable<? extends Number> getComparable(String numberExpression);

	private Tester getTester() {
		final Assume assumption = parameter.getAnnotation(Assume.class);
		if (assumption == null)
			return new TrueTester();

		final String expression = assumption.value();
		return getTester(expression);
	}

	protected Tester getTester(final String expression) {
		final Matcher matcher = EXPRESSION.matcher(expression);
		if (!matcher.matches())
			throw new TwipConfigurationErrorIllegalAssumeExpression("illegal assume expression ["
					+ expression + "] for [" + parameter.getType().getSimpleName() + "]");
		@SuppressWarnings("unchecked")
		final Comparable<Object> comparable = (Comparable) getComparable(matcher.group(3));
		if (">".equals(matcher.group(1))) {
			if ("=".equals(matcher.group(2))) {
				return new GreaterOrEqualsTester(comparable);
			} else {
				return new GreaterThanTester(comparable);
			}
		} else if ("<".equals(matcher.group(1))) {
			if ("=".equals(matcher.group(2))) {
				return new LessOrEqualsTester(comparable);
			} else {
				return new LessThanTester(comparable);
			}
		} else {
			if ("=".equals(matcher.group(2))) {
				return new EqualsTester(comparable);
			} else {
				throw new TwipConfigurationErrorIllegalAssumeExpression("illegal assume expression ["
						+ expression + "] for [" + parameter.getType().getSimpleName() + "]");
			}
		}
	}

	@Override
	public boolean test(Object value) {
		return tester.test(value);
	}
}
