package net.sf.twip.internal;

import static org.junit.Assert.assertEquals;

import java.lang.reflect.Method;
import java.util.Arrays;

import org.junit.runners.model.FrameworkMethod;

/**
 * A {@link FrameworkMethod} that knows about a fixed set of parameters passed
 * into {@link FrameworkMethod#invokeExplosively(Object, Object...)}
 */
public class FrameworkMethodWithArgs extends FrameworkMethod {

	private final Object[] parameters;

	private final String description;

	/**
	 * Copy-Constructor that appends one parameter.
	 */
	public FrameworkMethodWithArgs(FrameworkMethodWithArgs old, Object parameter) {
		super(old.getMethod());
		final int l = old.getName().length();
		this.description = old.getDescription().substring(0, l) + "." + toString(parameter)
				+ old.getDescription().substring(l);
		this.parameters = insertParameter(old.parameters, parameter);
	}

	/**
	 * Constructor that takes a method, defines the description to be its name,
	 * and define no parameters; you'll have to call the other constructor to
	 * derive a new instance with additional parameters, or invokeExplosively
	 * will fail.
	 */
	public FrameworkMethodWithArgs(Method method) {
		super(method);
		this.description = method.getName();
		this.parameters = new Object[0];
	}

	public String getDescription() {
		return description;
	}

	/** @return a new array with this one element added as the first element. */
	private Object[] insertParameter(Object[] oldParameters, Object newParameter) {
		final Object[] parameters = new Object[oldParameters.length + 1];
		System.arraycopy(oldParameters, 0, parameters, 1, oldParameters.length);
		parameters[0] = newParameter;
		return parameters;
	}

	@Override
	public Object invokeExplosively(Object target, Object... parameters) throws Throwable {
		assertEquals("don't call this with parameters; I know them already!", parameters.length, 0);
		assertEquals("you didn't specify all parameters for the method",
				getMethod().getParameterTypes().length, this.parameters.length);
		return super.invokeExplosively(target, this.parameters);
	}

	@Override
	public String toString() {
		return description + Arrays.toString(parameters);
	}

	/**
	 * @return the toString() of a parameter suitable as a method name suffix,
	 *         i.e. with returns and unicode characters replaced by harmless
	 *         escape sequences.
	 */
	private String toString(Object parameter) {
		final StringBuilder out = new StringBuilder();
		for (final char c : String.valueOf(parameter).toCharArray()) {
			switch (c) {
			case '\n':
				out.append("\\n");
				break;
			case '\r':
				out.append("\\r");
				break;
			case '\t':
				out.append("\\t");
				break;
			case '(':
				out.append("[");
				break;
			case ')':
				out.append("]");
				break;
			default:
				if (c > 256) {
					out.append("\\u");
					out.append(String.format("%04X", Integer.valueOf(c)));
				} else {
					out.append(c);
				}
			}
		}
		return out.toString();
	}

}