package net.sf.twip;

import java.lang.annotation.Annotation;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import net.sf.twip.internal.FailedFrameworkMethod;
import net.sf.twip.internal.FrameworkMethodWithArgs;
import net.sf.twip.internal.TwipConfigurationError;
import net.sf.twip.internal.TwipConfigurationErrorCantAccessExtension;
import net.sf.twip.internal.TwipConfigurationErrorCantInstantiateExtension;
import net.sf.twip.parameterhandler.AutoTwipParameterHandler;
import net.sf.twip.parameterhandler.BooleanParameterHandler;
import net.sf.twip.parameterhandler.ByteParameterHandler;
import net.sf.twip.parameterhandler.CharacterParameterHandler;
import net.sf.twip.parameterhandler.DoubleParameterHandler;
import net.sf.twip.parameterhandler.EnumParameterHandler;
import net.sf.twip.parameterhandler.FloatParameterHandler;
import net.sf.twip.parameterhandler.IntegerParameterHandler;
import net.sf.twip.parameterhandler.LongParameterHandler;
import net.sf.twip.parameterhandler.ParameterHandler;
import net.sf.twip.parameterhandler.ShortParameterHandler;
import net.sf.twip.parameterhandler.StringParameterHandler;
import net.sf.twip.util.Parameter;
import net.sf.twip.util.ReverseIterable;
import net.sf.twip.util.ServiceLoaderWrapper;
import net.sf.twip.util.ServiceLoadingException;

import org.junit.Test;
import org.junit.runners.BlockJUnit4ClassRunner;
import org.junit.runners.model.FrameworkMethod;
import org.junit.runners.model.InitializationError;
import org.junit.runners.model.Statement;

/**
 * "Tests with Properties" allows you to simply add parameters to your JUnit
 * tests. TwiP calls your test with all possible combinations of these
 * parameters... or at least some of them in the case of Integers, etc.
 * 
 * @see "http://twip.sf.net"
 */
public class TwiP extends BlockJUnit4ClassRunner {

	/** Enable the native Java <code>assert</code> keyword for all tests. */
	static {
		ClassLoader.getSystemClassLoader().setDefaultAssertionStatus(true);
	}

	/**
	 * @return the list of {@link FrameworkMethodWithArgs} that need to be
	 *         called for this method.
	 */
	public static List<FrameworkMethodWithArgs> computeParameterizedTestMethods(Method method) {
		List<FrameworkMethodWithArgs> result = new ArrayList<FrameworkMethodWithArgs>();
		result.add(new FrameworkMethodWithArgs(method));
		for (final Parameter parameter : ReverseIterable.create(Parameter.of(method))) {
			final List<FrameworkMethodWithArgs> next = new ArrayList<FrameworkMethodWithArgs>();
			final ParameterHandler parameterHandler = getParameterHandler(parameter);
			for (final Object value : parameterHandler.getParameterValues()) {
				for (final FrameworkMethodWithArgs old : result) {
					next.add(new FrameworkMethodWithArgs(old, value));
				}
			}
			result = next;
		}
		return result;
	}

	private static ParameterHandler getParameterHandler(Parameter parameter) {
		final Class<?> type = parameter.getType();
		if (type.isEnum()) {
			return new EnumParameterHandler(parameter);
		} else if (Boolean.class.isAssignableFrom(type) || Boolean.TYPE.equals(type)) {
			return new BooleanParameterHandler(parameter);
		} else if (Byte.class.isAssignableFrom(type) || Byte.TYPE.equals(type)) {
			return new ByteParameterHandler(parameter);
		} else if (Character.class.isAssignableFrom(type) || Character.TYPE.equals(type)) {
			return new CharacterParameterHandler(parameter);
		} else if (Double.class.isAssignableFrom(type) || Double.TYPE.equals(type)) {
			return new DoubleParameterHandler(parameter);
		} else if (Float.class.isAssignableFrom(type) || Float.TYPE.equals(type)) {
			return new FloatParameterHandler(parameter);
		} else if (Integer.class.isAssignableFrom(type) || Integer.TYPE.equals(type)) {
			return new IntegerParameterHandler(parameter);
		} else if (Long.class.isAssignableFrom(type) || Long.TYPE.equals(type)) {
			return new LongParameterHandler(parameter);
		} else if (Short.class.isAssignableFrom(type) || Short.TYPE.equals(type)) {
			return new ShortParameterHandler(parameter);
		} else if (String.class.isAssignableFrom(type)) {
			return new StringParameterHandler(parameter);
		} else {
			return new AutoTwipParameterHandler(parameter);
		}
	}

	private List<FrameworkMethod> testMethods;

	private final List<TwipExtension> extensions = new ArrayList<TwipExtension>();

	public TwiP(Class<?> testClass) throws InitializationError {
		super(testClass);
		initExtensions(testClass);
	}

	@Override
	protected List<FrameworkMethod> computeTestMethods() {
		if (testMethods == null) {
			testMethods = new ArrayList<FrameworkMethod>();
			for (final FrameworkMethod method : getTestClass().getAnnotatedMethods(Test.class)) {
				if (method.getMethod().getParameterTypes().length == 0) {
					testMethods.add(method);
					continue;
				}
				List<? extends FrameworkMethod> parameterizedTestMethods;
				try {
					parameterizedTestMethods = computeParameterizedTestMethods(method.getMethod());
				} catch (final TwipConfigurationError e) {
					parameterizedTestMethods = createFailedMethod(method, e);
				}
				testMethods.addAll(parameterizedTestMethods);
			}
		}
		return testMethods;
	}

	private List<FailedFrameworkMethod> createFailedMethod(final FrameworkMethod method, final Throwable e) {
		return Collections.singletonList(new FailedFrameworkMethod(method.getMethod(), e));
	}

	private void initExtensions(Class<?> testClass) throws InitializationError {
		final List<Throwable> errors = new ArrayList<Throwable>();

		final Set<Class<? extends TwipExtension>> loaded = instantiateServiceLoaderExtensions(errors);

		instantiateAnnotatedExtensions(testClass, loaded, errors);

		if (!errors.isEmpty()) {
			throw new InitializationError(errors);
		}
	}

	private void instantiateAnnotatedExtensions(Class<?> testClass,
			Set<Class<? extends TwipExtension>> loaded, final List<Throwable> errors) {
		final TwipExtensions annotation = testClass.getAnnotation(TwipExtensions.class);
		if (annotation != null) {
			for (final Class<? extends TwipExtension> type : annotation.value()) {
				if (loaded.contains(type))
					continue;
				try {
					extensions.add(type.newInstance());
				} catch (final InstantiationException e) {
					final String message = "can't instantiate " + type;
					errors.add(new TwipConfigurationErrorCantInstantiateExtension(message, e));
				} catch (final IllegalAccessException e) {
					errors.add(new TwipConfigurationErrorCantAccessExtension("can't access " + type, e));
				}
			}
		}
	}

	private Set<Class<? extends TwipExtension>> instantiateServiceLoaderExtensions(List<Throwable> errors) {
		final Set<Class<? extends TwipExtension>> serviceLoadedTypes = new HashSet<Class<? extends TwipExtension>>();
		final Iterator<TwipExtension> loaders = ServiceLoaderWrapper.load(TwipExtension.class).iterator();
		while (loaders.hasNext()) {
			try {
				final TwipExtension extension = loaders.next();
				extensions.add(extension);
				serviceLoadedTypes.add(extension.getClass());
			} catch (final ServiceLoadingException e) {
				errors.add(e);
			}
		}
		return serviceLoadedTypes;
	}

	@Override
	protected Statement methodInvoker(FrameworkMethod method, final Object testInstance) {
		Statement statement = super.methodInvoker(method, testInstance);
		for (final TwipExtension extension : extensions) {
			statement = extension.wrap(true, method, testInstance, statement);
		}
		return statement;
	}

	@Override
	protected String testName(FrameworkMethod method) {
		return (method instanceof FrameworkMethodWithArgs) //
		? ((FrameworkMethodWithArgs) method).getDescription()
				: super.testName(method);
	}

	@Override
	protected void validatePublicVoidNoArgMethods(Class<? extends Annotation> annotation, boolean isStatic,
			List<Throwable> errors) {
		final List<FrameworkMethod> methods = getTestClass().getAnnotatedMethods(annotation);
		for (final FrameworkMethod eachTestMethod : methods) {
			eachTestMethod.validatePublicVoid(isStatic, errors);
		}
	}

	@Override
	protected Statement withAfters(FrameworkMethod method, final Object testInstance, Statement statement) {
		Statement result = super.withAfters(method, testInstance, statement);
		for (final TwipExtension extension : extensions) {
			result = extension.wrap(false, method, testInstance, result);
		}
		return result;
	}
}
